<?php

declare(strict_types=1);

/*
 * This file is part of the RegexParser package.
 *
 * (c) Younes ENNAJI <younes.ennaji.pro@gmail.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace RegexParser\Bridge\Symfony\DependencyInjection;

use RegexParser\Regex;
use Symfony\Component\Config\Definition\Builder\TreeBuilder;
use Symfony\Component\Config\Definition\ConfigurationInterface;

/**
 * Defines the configuration schema for the RegexParser bundle.
 */
final readonly class Configuration implements ConfigurationInterface
{
    /**
     * @return TreeBuilder<'array'> the tree builder instance
     */
    public function getConfigTreeBuilder(): TreeBuilder
    {
        $treeBuilder = new TreeBuilder('regex_parser');

        $treeBuilder->getRootNode()
            ->children()
                ->integerNode('max_pattern_length')
                    ->defaultValue(Regex::DEFAULT_MAX_PATTERN_LENGTH)
                    ->info('The maximum allowed length for a regex pattern string to parse.')
                ->end()
                ->integerNode('max_lookbehind_length')
                    ->defaultValue(Regex::DEFAULT_MAX_LOOKBEHIND_LENGTH)
                    ->min(0)
                    ->info('The maximum allowed lookbehind length. Can be overridden per-pattern via (*LIMIT_LOOKBEHIND=...).')
                ->end()
                ->booleanNode('runtime_pcre_validation')
                    ->defaultValue('%kernel.debug%')
                    ->info('Whether to validate patterns against the runtime PCRE engine (preg_match compile check).')
                ->end()
                ->arrayNode('cache')
                    ->addDefaultsIfNotSet()
                    ->info('Cache configuration for storing parsed regex patterns.')
                    ->children()
                        ->scalarNode('pool')
                            ->defaultNull()
                            ->info('Symfony cache pool service id (PSR-6). Takes precedence over "directory" when set.')
                        ->end()
                         ->scalarNode('directory')
                             ->defaultValue('%kernel.cache_dir%/regex_parser')
                             ->info('Directory path for cached AST files. Set to null to disable caching.')
                        ->end()
                        ->scalarNode('prefix')
                            ->defaultValue('regex_')
                            ->info('Cache key prefix for PSR-6 cache pools.')
                        ->end()
                    ->end()
                ->end()
                ->scalarNode('extractor_service')
                    ->defaultNull()
                    ->info('Custom regex pattern extractor service ID. If not provided, PhpParser-based extraction will be tried first, then token-based extraction.')
                ->end()
                ->arrayNode('redos')
                    ->addDefaultsIfNotSet()
                    ->children()
                        ->scalarNode('threshold')
                            ->defaultValue('high')
                            ->info('Minimum ReDoS severity to report (safe|low|medium|high|critical).')
                            ->beforeNormalization()
                                ->ifString()
                                ->then(static fn (string $value): string => strtolower($value))
                            ->end()
                            ->validate()
                                ->ifNotInArray(['safe', 'low', 'medium', 'high', 'critical'])
                                ->thenInvalid('Invalid "regex_parser.redos.threshold" value "%s". Allowed: safe, low, medium, high, critical.')
                            ->end()
                        ->end()
                        ->arrayNode('ignored_patterns')
                            ->scalarPrototype()->end()
                            ->defaultValue([])
                            ->info('List of patterns or full regexes to exclude from ReDoS analysis.')
                        ->end()
                    ->end()
                ->end()
                ->arrayNode('analysis')
                    ->addDefaultsIfNotSet()
                    ->children()
                        ->integerNode('warning_threshold')
                            ->defaultValue(50)
                            ->min(0)
                            ->info('Complexity score above which a warning is emitted.')
                        ->end()
                        ->integerNode('redos_threshold')
                            ->defaultValue(100)
                            ->min(0)
                            ->info('Complexity score above which a pattern is flagged as ReDoS risk.')
                        ->end()
                        ->arrayNode('ignore_patterns')
                            ->scalarPrototype()->end()
                            ->defaultValue([])
                            ->info('List of regex fragments to treat as safe (e.g. Symfony requirement constants).')
                        ->end()
                    ->end()
                ->end()
                ->arrayNode('automata')
                    ->addDefaultsIfNotSet()
                    ->children()
                        ->scalarNode('minimization_algorithm')
                            ->defaultValue('hopcroft')
                            ->info('DFA minimization strategy for automata comparisons (hopcroft|moore).')
                            ->beforeNormalization()
                                ->ifString()
                                ->then(static fn (string $value): string => strtolower($value))
                            ->end()
                            ->validate()
                                ->ifNotInArray(['hopcroft', 'moore'])
                                ->thenInvalid('Invalid "regex_parser.automata.minimization_algorithm" value "%s". Allowed: hopcroft, moore.')
                            ->end()
                        ->end()
                        ->scalarNode('determinization_algorithm')
                            ->defaultValue('subset-indexed')
                            ->info('NFA determinization strategy for automata comparisons (subset|subset-indexed).')
                            ->beforeNormalization()
                                ->ifString()
                                ->then(static fn (string $value): string => strtolower($value))
                            ->end()
                            ->validate()
                                ->ifNotInArray(['subset', 'subset-indexed'])
                                ->thenInvalid('Invalid "regex_parser.automata.determinization_algorithm" value "%s". Allowed: subset, subset-indexed.')
                            ->end()
                        ->end()
                    ->end()
                ->end()
                ->arrayNode('optimizations')
                    ->addDefaultsIfNotSet()
                    ->info('Default optimization options for regex:lint.')
                    ->children()
                        ->booleanNode('digits')
                            ->defaultTrue()
                            ->info('Optimize digit character classes (e.g., [0-9] -> \\d).')
                        ->end()
                        ->booleanNode('word')
                            ->defaultTrue()
                            ->info('Optimize word character classes (e.g., [A-Za-z0-9_] -> \\w).')
                        ->end()
                        ->booleanNode('ranges')
                            ->defaultTrue()
                            ->info('Allow range merging inside character classes.')
                        ->end()
                        ->booleanNode('canonicalize_char_classes')
                            ->defaultTrue()
                            ->info('Normalize character class order and deduplicate elements.')
                        ->end()
                        ->booleanNode('possessive')
                            ->defaultFalse()
                            ->info('Enable auto-possessive quantifier optimizations.')
                        ->end()
                        ->booleanNode('factorize')
                            ->defaultFalse()
                            ->info('Enable alternation factorization optimizations.')
                        ->end()
                        ->integerNode('min_quantifier_count')
                            ->defaultValue(4)
                            ->min(2)
                            ->info('Minimum repeated quantifier count before collapsing (e.g., aaaa -> a{4}).')
                        ->end()
                    ->end()
                ->end()
                ->arrayNode('paths')
                    ->scalarPrototype()->end()
                    ->defaultValue(['src'])
                    ->info('Directories to scan for regex patterns. Defaults to src/ for Symfony applications.')
                ->end()
                ->arrayNode('exclude_paths')
                    ->scalarPrototype()->end()
                    ->defaultValue(['vendor'])
                    ->info('Directories to exclude from scanning. Defaults to vendor/, tests/, and Fixtures/ for Symfony applications.')
                ->end()
                ->scalarNode('ide')
                    ->defaultValue('%env(default::SYMFONY_IDE)%')
                    ->info('IDE shorthand (vscode, phpstorm, etc.) or custom URL template for clickable links (e.g., phpstorm://open?file=%%file%%&line=%%line%%&column=%%column%%). Falls back to framework.ide.')
                ->end()
            ->end();

        return $treeBuilder;
    }
}
